require("data.classes.Collection")
require("data.classes.Factory")
require("data.classes.Subfactory")
require("data.classes.Floor")
require("data.classes.Line")
require("data.classes.Recipe")
require("data.classes.Machine")
require("data.classes.Beacon")
require("data.classes.ModuleSet")
require("data.classes.Module")
require("data.classes.Item")
require("data.classes.Fuel")

require("data.handlers.generator")
require("data.handlers.loader")
require("data.handlers.migrator")
require("data.handlers.prototyper")
require("data.handlers.screenshotter")

require("data.calculation.interface")

-- ** LOCAL UTIL **
local function reload_settings(player)
    -- Writes the current user mod settings to their player_table, for read-performance
    local settings = settings.get_player_settings(player)
    local settings_table = {}

    local timescale_to_number = {one_second = 1, one_minute = 60, one_hour = 3600}

    settings_table.show_gui_button = settings["fp_display_gui_button"].value
    settings_table.products_per_row = tonumber(settings["fp_products_per_row"].value)
    settings_table.subfactory_list_rows = tonumber(settings["fp_subfactory_list_rows"].value)
    settings_table.default_timescale = timescale_to_number[settings["fp_default_timescale"].value]
    settings_table.belts_or_lanes = settings["fp_view_belts_or_lanes"].value
    settings_table.prefer_product_picker = settings["fp_prefer_product_picker"].value
    settings_table.prefer_matrix_solver = settings["fp_prefer_matrix_solver"].value

    global.players[player.index].settings = settings_table
end

local function reload_preferences(player)
    -- Reloads the user preferences, incorporating previous preferences if possible
    local preferences = global.players[player.index].preferences

    preferences.pause_on_interface = preferences.pause_on_interface or false
    preferences.fold_out_subfloors = preferences.fold_out_subfloors or false
    preferences.show_floor_items = preferences.show_floor_items or false
    preferences.tutorial_mode = preferences.tutorial_mode or true
    preferences.utility_scopes = preferences.utility_scopes or {components = "Subfactory"}
    preferences.recipe_filters = preferences.recipe_filters or {disabled = false, hidden = false}

    preferences.ignore_barreling_recipes = preferences.ignore_barreling_recipes or false
    preferences.ignore_recycling_recipes = preferences.ignore_recycling_recipes or false
    preferences.ingredient_satisfaction = preferences.ingredient_satisfaction or false
    preferences.round_button_numbers = preferences.round_button_numbers or false
    preferences.attach_subfactory_products = preferences.attach_subfactory_products or false

    preferences.done_column = preferences.done_column or false
    preferences.pollution_column = preferences.pollution_column or false
    preferences.line_comment_column = preferences.line_comment_column or false

    preferences.mb_defaults = preferences.mb_defaults or
      {machine = nil, machine_secondary = nil, beacon = nil, beacon_count = nil}

    preferences.default_prototypes = preferences.default_prototypes or {}
    preferences.default_prototypes = {
        belts = preferences.default_prototypes.belts or prototyper.defaults.get_fallback("belts"),
        beacons = preferences.default_prototypes.beacons or prototyper.defaults.get_fallback("beacons"),
        wagons = preferences.default_prototypes.wagons or prototyper.defaults.get_fallback("wagons"),
        fuels = preferences.default_prototypes.fuels or prototyper.defaults.get_fallback("fuels"),
        machines = preferences.default_prototypes.machines or prototyper.defaults.get_fallback("machines")
    }
end

local function reset_ui_state(player)
    local ui_state_table = {}

    ui_state_table.main_dialog_dimensions = nil  -- Can only be calculated after on_init
    ui_state_table.last_action = nil  -- The last user action (used for rate limiting)
    ui_state_table.view_states = nil  -- The state of the production views
    ui_state_table.message_queue = {}  -- The general message queue
    ui_state_table.main_elements = {}  -- References to UI elements in the main interface
    ui_state_table.compact_elements = {}  -- References to UI elements in the compact interface
    ui_state_table.context = ui_util.context.create(player)  -- The currently displayed set of data
    ui_state_table.last_selected_picker_group = nil  -- The item picker category that was last selected

    ui_state_table.modal_dialog_type = nil  -- The internal modal dialog type
    ui_state_table.modal_data = nil  -- Data that can be set for a modal dialog to use
    ui_state_table.queued_dialog_settings = nil  -- Info on dialog to open after the current one closes

    ui_state_table.flags = {
        archive_open = false,  -- Wether the players subfactory archive is currently open
        selection_mode = false,  -- Whether the player is currently using a selector
        compact_view = false,  -- Whether the user has switched to the compact main view
        recalculate_on_subfactory_change = false  -- Whether calculations should re-run
    }

    -- The UI table gets replaced because the whole interface is reset
    global.players[player.index].ui_state = ui_state_table
end


-- Makes sure that the given player has a player_table and a reset gui state
local function update_player_table(player)
    local function reload_data()
        reload_settings(player)  -- reloads the settings of the player
        reload_preferences(player) -- reloads and adjusts the player's preferences
        reset_ui_state(player)  -- Resets the player's UI state
    end

    local player_table = global.players[player.index]
    if player_table == nil then  -- new player
        global.players[player.index] = {}
        player_table = global.players[player.index]

        player_table.mod_version = global.mod_version
        player_table.index = player.index

        player_table.factory = Factory.init()
        player_table.archive = Factory.init()

        player_table.settings = {}
        player_table.preferences = {}
        player_table.ui_state = {}
        reload_data()

        title_bar.enqueue_message(player, {"fp.hint_tutorial"}, "hint", 5, false)

    else  -- existing player, only needs to update
        reload_data()

        local archive_subfactories = Factory.get_in_order(player_table.archive, "Subfactory")
        if next(archive_subfactories) then player_table.archive.selected_subfactory = archive_subfactories[1] end

        local factory = player_table.factory
        local subfactories = Factory.get_in_order(factory, "Subfactory")
        if next(subfactories) then
            local subfactory_to_select = subfactories[1]
            if factory.selected_subfactory ~= nil then
                -- Get the selected subfactory from the factory to make sure it still exists
                local selected_subfactory = Factory.get(factory, "Subfactory", factory.selected_subfactory.id)
                if selected_subfactory ~= nil then subfactory_to_select = selected_subfactory end
            end
            ui_util.context.set_subfactory(player, subfactory_to_select)
        end
    end

    -- Translation tables and clipboard are re-initialized every time
    player_table.translation_tables = nil
    player_table.clipboard = nil

    return player_table
end


-- Downscale width and height mod settings until the main interface fits onto the player's screen
function NTH_TICK_HANDLERS.shrinkwrap_interface(metadata)
    local player = game.get_player(metadata.player_index)
    local resolution, scale = player.display_resolution, player.display_scale
    local actual_resolution = {width=math.ceil(resolution.width / scale), height=math.ceil(resolution.height / scale)}

    local mod_settings = data_util.get("settings", player)
    local products_per_row = mod_settings.products_per_row
    local subfactory_list_rows = mod_settings.subfactory_list_rows

    local function determine_dimensions()
        return main_dialog.determine_main_dialog_dimensions(player, products_per_row, subfactory_list_rows)
    end

    while (actual_resolution.width * 0.95) < determine_dimensions().width do
        products_per_row = products_per_row - 1
    end
    while (actual_resolution.height * 0.95) < determine_dimensions().height do
        subfactory_list_rows = subfactory_list_rows - 2
    end

    local setting_prototypes = game.mod_setting_prototypes
    local width_minimum = setting_prototypes["fp_products_per_row"].allowed_values[1]
    local height_minimum = setting_prototypes["fp_subfactory_list_rows"].allowed_values[1]

    local live_settings = settings.get_player_settings(player)
    live_settings["fp_products_per_row"] = {value = math.max(products_per_row, width_minimum)}
    live_settings["fp_subfactory_list_rows"] = {value = math.max(subfactory_list_rows, height_minimum)}
end


local function global_init()
    -- Set up a new save for development if necessary
    local freeplay = remote.interfaces["freeplay"]
    if DEVMODE and freeplay then  -- Disable freeplay popup-message
        if freeplay["set_skip_intro"] then remote.call("freeplay", "set_skip_intro", true) end
        if freeplay["set_disable_crashsite"] then remote.call("freeplay", "set_disable_crashsite", true) end
    end

    -- Initiates all factorio-global variables
    global.mod_version = game.active_mods["factoryplanner"]
    global.players = {}

    -- Save metadata about currently registered on_nth_tick events
    global.nth_tick_events = {}

    -- Run through the prototyper without the need to apply (run) it on any player
    global.tutorial_subfactory_validity = nil
    global.installed_mods = nil

    prototyper.setup()
    prototyper.finish()

    -- Initialize flib's translation module
    translator.on_init()
    prototyper.util.build_translation_dictionaries()

    -- Create player tables for all existing players
    for _, player in pairs(game.players) do update_player_table(player) end
end

-- Prompts migrations, a GUI and prototype reload, and a validity check on all subfactories
local function handle_configuration_change()
    prototyper.setup()  -- Setup prototyper
    migrator.migrate_global()  -- Migrate global

    -- Runs through all players, even new ones without player_table
    for _, player in pairs(game.players) do
        -- Migrate player_table data if it exists
        migrator.migrate_player_table(player)

        -- Create or update player_table
        local player_table = update_player_table(player)

        -- Migrate the default prototypes for the player
        prototyper.run(player_table)

        -- Update the validity of the entire factory and archive
        Collection.validate_datasets(player_table.factory.Subfactory, Subfactory)
        Collection.validate_datasets(player_table.archive.Subfactory, Subfactory)
    end

    -- Complete prototyper process by saving new data to global
    prototyper.finish()

    -- Re-initialize flib's translation module
    translator.on_configuration_changed()
    prototyper.util.build_translation_dictionaries()

    for index, player in pairs(game.players) do
        ui_util.reset_player_gui(player)  -- Destroys all existing GUI's
        ui_util.toggle_mod_gui(player)  -- Recreates the mod-GUI if necessary

        -- Update factory and archive calculations in case prototypes changed in a relevant way
        local player_table = global.players[index]
        for _, factory_name in pairs{"factory", "archive"} do
            for _, subfactory in ipairs(Factory.get_in_order(player_table[factory_name], "Subfactory")) do
                calculation.update(player, subfactory)
            end
        end
    end
end


-- ** TOP LEVEL **
script.on_init(global_init)

script.on_configuration_changed(handle_configuration_change)

script.on_load(loader.run)


-- ** PLAYER DATA **
script.on_event(defines.events.on_player_created, function(event)
    local player = game.get_player(event.player_index)

    -- Sets up the player_table for the new player
    update_player_table(player)

    -- Sets up the mod-GUI for the new player if necessary
    ui_util.toggle_mod_gui(player)

    -- Make sure the width and height mod settings are appropriate
    -- Resolution and scale are not loaded for the player at this point, so we need to delay this action a tick
    data_util.nth_tick.add((game.tick + 1), "shrinkwrap_interface", {player_index=player.index})

    -- Add the subfactories that are handy for development
    if DEVMODE then data_util.add_subfactories_by_string(player, DEV_EXPORT_STRING) end
end)

script.on_event(defines.events.on_player_removed, function(event)
    global.players[event.player_index] = nil
end)


script.on_event(defines.events.on_player_joined_game, translator.on_player_joined_game)

script.on_event(defines.events.on_runtime_mod_setting_changed, function(event)
    if event.setting_type == "runtime-per-user" then  -- this mod only has per-user settings
        local player = game.get_player(event.player_index)
        reload_settings(player)

        if event.setting == "fp_display_gui_button" then
            ui_util.toggle_mod_gui(player)

        elseif event.setting == "fp_products_per_row" or
          event.setting == "fp_subfactory_list_rows" or
          event.setting == "fp_prefer_product_picker" then
            main_dialog.rebuild(player, false)

        elseif event.setting == "fp_view_belts_or_lanes" then
            local player_table = data_util.get("table", player)

            -- Goes through every subfactory's top level products and updates their defined_by
            local defined_by = player_table.settings.belts_or_lanes
            Factory.update_product_definitions(player_table.factory, defined_by)
            Factory.update_product_definitions(player_table.archive, defined_by)
            local subfactory = player_table.ui_state.context.subfactory

            calculation.update(player, subfactory)
            main_dialog.rebuild(player, false)

        end
    end
end)


-- ** TRANSLATION **
-- Required by flib's translation module
script.on_event(defines.events.on_tick, translator.on_tick)

-- Keep translation going
script.on_event(defines.events.on_string_translated, translator.on_string_translated)

-- Save translations once they are complete
script.on_event(translator.on_player_dictionaries_ready, function(event)
  local player = game.get_player(event.player_index)
  local player_table = data_util.get("table", player)

  player_table.translation_tables = translator.get_all(event.player_index)
  modal_dialog.set_searchfield_state(player)  -- enables searchfields if possible
end)


-- ** COMMANDS **
commands.add_command("fp-reset-prototypes", {"command-help.fp_reset_prototypes"}, handle_configuration_change)
commands.add_command("fp-restart-translation", {"command-help.fp_restart_translation"}, function()
    translator.on_init()
    prototyper.util.build_translation_dictionaries()
end)
